function [sys,E]=Hydrog(Z,l,m,varargin)
% Etats confins d'un atome hydrognode
%   >> L            = largeur (nm) du puits
%   >> Ep0          = Profondeur (eV) du puits (Ep0 < 0)
%   >> m            = Masse (par rapport  l'lectron) de l'objet physique
%   >> varargin     = Arguments d'entre optionnels
%           >> nx   = Nombre de point selon Ox (dfaut : 200)
%           >> nE   = Nombre de valeurs de l'nergie (dfaut : 1000)
%           >> Emin = Valeur minimale de l'nergie (dfaut : min(Ep))
%           >> Emax = Valeur maximale de l'nergie (dfaut : 0)
%
% Olivier Pujol : septembre 2013

%% Vrifications

if (l==0),error('Cet exemple ne fonctionne que pour l > 0');end
if (Z<=0),error('On doit avoir Z >= 1');end

%% Arguments d'entre optionnels

switch length(varargin)
    case 0
        nx=200;                     % Valeur par dfaut du nombre de points selon Ox
        nE=1000;                    % Valeur par dfaut du nombre de valeurs de l'nergie
    case 1
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=1000;                    % Valeur par dfaut du nombre de valeurs de l'nergie
    case 2
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=varargin{2};             % Valeur par dfaut du nombre de valeurs de l'nergie
    case 4
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=varargin{2};             % Valeur par dfaut du nombre de valeurs de l'nergie
        E=linspace(varargin{3},varargin{4},nE);              % Ensemble des valeurs de l'nergie
    otherwise
        ('Mauvais nombre d''arguments d''entre optionnels');
end

%% Energie potentielle

% Constantes
Ry = 13.6058;                       % Rydberg (eV)
aB = 0.0529177;                     % Rayon de Bohr (nm)

Ep0 = Z^2*Ry/(l*(l+1));             % Minimum de l'nergie potentielle
x0 = l*(l+1)*aB/Z;                  % Point pour lequel l'nergie potentielle est minimale

% Construction
x=linspace(x0/4,50*x0,nx);          % Domaine spatial
Epc=2*Ep0.*(-(x0./x)+(x0^2./(2.*x.^2)));     % Energie potentielle coulombienne + centrifuge (origine en haut du puits)

% Discrtisation
[M,xt,Ep]=discretisation(x,Epc); 
axis([0 50*x0 min(Epc) 0]);

clear Ry aB Ep0 Epc x

% Valeurs de l'nergie
if (length(varargin)~=4)
    E=linspace(min(Ep),Ep(end),nE);   % Ensemble des valeurs de l'nergie
end    

if (max(E)>0),error('Certaines valeurs de E sont hors du domaine des tats confins');end

%% Etats confins

[sys,E]=Epot(M,xt,Ep,m,E);          % Fabrication du systme 
sysconf(sys,E);                     % Etats confins

return;